<?php
/**
 * Plugin Name: Wallet Transactions Editor
 * Description: Allows admin to view, search, edit, and delete records from the `wpew_woo_wallet_transactions` table.
 * Version: 2.0
 * Author: Your Name
 */

if (!defined('ABSPATH')) {
    exit; // Prevent direct access
}

// Add admin menu
function wte_add_admin_menu() {
    add_menu_page(
        'Wallet Transactions',
        'Wallet Transactions',
        'manage_options',
        'wallet-transactions',
        'wte_admin_page',
        'dashicons-money-alt',
        20
    );
}
add_action('admin_menu', 'wte_add_admin_menu');

// Enqueue admin JS for inline editing
function wte_enqueue_admin_scripts($hook) {
    if ($hook !== 'toplevel_page_wallet-transactions') return;
    
    wp_enqueue_script('wte-admin-js', plugin_dir_url(__FILE__) . 'wte-admin.js', array('jquery', 'jquery-ui-datepicker'), '1.1', true);
    wp_enqueue_style('jquery-ui-css', 'https://ajax.googleapis.com/ajax/libs/jqueryui/1.12.1/themes/smoothness/jquery-ui.css');
    
    wp_localize_script('wte-admin-js', 'wte_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('wte_ajax_nonce'),
    ));
    
    // Output the JavaScript directly if we can't create a separate file
    add_action('admin_footer', 'wte_output_js');
}
add_action('admin_enqueue_scripts', 'wte_enqueue_admin_scripts');

// Output JavaScript directly in the footer
function wte_output_js() {
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        // Handle update button clicks
        $(document).on('click', '.wte-update-btn', function() {
            var $row = $(this).closest('tr');
            var transaction_id = $row.data('id');
            var $msg = $row.find('.wte-msg');
            
            $msg.text('Updating...').css('color', 'blue');
            
            var data = {
                action: 'wte_update_transaction',
                nonce: wte_ajax.nonce,
                transaction_id: transaction_id,
                amount: $row.find('.wte-amount').val(),
                type: $row.find('.wte-type').val(),
                details: $row.find('.wte-details').val(),
                date: $row.find('.wte-date').val()
            };
            
            $.post(wte_ajax.ajax_url, data, function(response) {
                if (response.success) {
                    $msg.text(response.data).css('color', 'green');
                    setTimeout(function() { $msg.text(''); }, 3000);
                } else {
                    $msg.text(response.data).css('color', 'red');
                }
            }).fail(function() {
                $msg.text('Error: Request failed').css('color', 'red');
            });
        });
        
        // Handle delete button clicks
        $(document).on('click', '.wte-delete-btn', function() {
            if (!confirm('Are you sure you want to delete this transaction?')) {
                return;
            }
            
            var $row = $(this).closest('tr');
            var transaction_id = $row.data('id');
            var $msg = $row.find('.wte-msg');
            
            $msg.text('Deleting...').css('color', 'blue');
            
            var data = {
                action: 'wte_delete_transaction',
                nonce: wte_ajax.nonce,
                transaction_id: transaction_id
            };
            
            $.post(wte_ajax.ajax_url, data, function(response) {
                if (response.success) {
                    $msg.text(response.data).css('color', 'green');
                    setTimeout(function() { $row.fadeOut(300, function() { $(this).remove(); }); }, 1000);
                } else {
                    $msg.text(response.data).css('color', 'red');
                }
            }).fail(function() {
                $msg.text('Error: Request failed').css('color', 'red');
            });
        });
        
        // Initialize date picker for datetime-local inputs
        $('.wte-date').each(function() {
            $(this).datepicker({
                dateFormat: 'yy-mm-dd',
                onSelect: function(dateText) {
                    var time = $(this).val().split('T')[1] || '00:00';
                    $(this).val(dateText + 'T' + time);
                }
            });
        });
        
        // Select all checkbox functionality
        $('#select-all').on('click', function() {
            var checkboxes = $("input[name='bulk_ids[]']");
            checkboxes.prop('checked', $(this).prop('checked'));
        });
    });
    </script>
    <?php
}

// AJAX handler for updating transaction
add_action('wp_ajax_wte_update_transaction', function() {
    check_ajax_referer('wte_ajax_nonce', 'nonce');
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    global $wpdb;
    $table_name = $wpdb->prefix . "woo_wallet_transactions";
    $id = intval($_POST['transaction_id']);
    $amount = floatval($_POST['amount']);
    $type = sanitize_text_field($_POST['type']);
    $details = sanitize_textarea_field($_POST['details']);
    $date = sanitize_text_field($_POST['date']);
    if (strpos($date, 'T') !== false) {
        $date = str_replace('T', ' ', $date);
    }
    $result = $wpdb->update(
        $table_name,
        ['amount' => $amount, 'type' => $type, 'details' => $details, 'date' => $date],
        ['transaction_id' => $id]
    );
    if ($result !== false) {
        wp_send_json_success('Transaction updated successfully.');
    } else {
        wp_send_json_error('Failed to update transaction.');
    }
});

// AJAX handler for deleting transaction
add_action('wp_ajax_wte_delete_transaction', function() {
    check_ajax_referer('wte_ajax_nonce', 'nonce');
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    global $wpdb;
    $table_name = $wpdb->prefix . "woo_wallet_transactions";
    $id = intval($_POST['transaction_id']);
    $result = $wpdb->delete($table_name, ['transaction_id' => $id]);
    if ($result !== false) {
        wp_send_json_success('Transaction deleted successfully.');
    } else {
        wp_send_json_error('Failed to delete transaction.');
    }
});

// Admin Page Content
function wte_admin_page() {
    global $wpdb;
    $table_name = $wpdb->prefix . "woo_wallet_transactions";

    // Pagination settings
    $limit = 50;
    $page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
    $offset = ($page - 1) * $limit;

    // Search functionality
    $search_query = isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '';

    $where_clause = '';
    if (!empty($search_query)) {
        $where_clause = $wpdb->prepare(
            "WHERE details LIKE %s OR amount LIKE %s OR type LIKE %s",
            "%{$search_query}%", "%{$search_query}%", "%{$search_query}%"
        );
    }

    // Handle record updates
    if (isset($_POST['update_transaction'])) {
        $id = intval($_POST['transaction_id']);
        $amount = floatval($_POST['amount']);
        $type = sanitize_text_field($_POST['type']);
        $details = sanitize_textarea_field($_POST['details']);
        
        // Handle date input (fix for datetime-local validation)
        $date = sanitize_text_field($_POST['date']);
        if (strpos($date, 'T') !== false) {
            $date = str_replace('T', ' ', $date); // Convert HTML5 format to MySQL format
        }

        $wpdb->update(
            $table_name,
            ['amount' => $amount, 'type' => $type, 'details' => $details, 'date' => $date],
            ['transaction_id' => $id]
        );

        echo "<div class='updated'><p>Transaction Updated Successfully!</p></div>";
    }

    // Handle record deletions
    if (isset($_POST['delete_transaction'])) {
        $id = intval($_POST['transaction_id']);
        $wpdb->delete($table_name, ['transaction_id' => $id]);

        echo "<div class='updated'><p>Transaction Deleted Successfully!</p></div>";
    }

    // Handle bulk deletions
    if (isset($_POST['bulk_apply']) && $_POST['bulk_action'] === 'delete' && !empty($_POST['bulk_ids'])) {
        $ids = array_map('intval', $_POST['bulk_ids']);
        $ids_placeholders = implode(',', array_fill(0, count($ids), '%d'));
        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE transaction_id IN ($ids_placeholders)",
            $ids
        ));
        echo "<div class='updated'><p>Selected transactions deleted successfully!</p></div>";
    }

    // Fetch total records for pagination
    $total_records = $wpdb->get_var("SELECT COUNT(*) FROM $table_name $where_clause");
    $total_pages = ceil($total_records / $limit);

    // Fetch paginated transactions with search filter
    $transactions = $wpdb->get_results("SELECT transaction_id, user_id, type, amount, details, date FROM $table_name $where_clause ORDER BY transaction_id DESC LIMIT $limit OFFSET $offset");

    // Table styling (remove most inline styles, use WP classes)
    echo '<div class="wrap"><h2>Wallet Transactions</h2>';

    // Bulk action and search form (GET for search, POST for bulk)
    echo '<form method="get" style="margin-bottom: 10px; display:inline-block;">';
    echo '<input type="hidden" name="page" value="wallet-transactions">';
    echo '<input type="text" name="search" value="' . esc_attr($search_query) . '" placeholder="Search transactions..." style="padding:5px; width:250px;">';
    echo '<input type="submit" value="Search" class="button button-primary">';
    echo '</form>';

    // Bulk action form
    echo '<form method="post" id="bulk-delete-form">';
    echo '<div class="tablenav top">';
    echo '<div class="alignleft actions bulkactions">';
    echo '<select name="bulk_action">
            <option value="">Bulk actions</option>
            <option value="delete">Delete</option>
          </select>';
    echo '<input type="submit" name="bulk_apply" id="doaction" class="button action" value="Apply" onclick="return confirm(\'Are you sure you want to delete the selected transactions?\');">';
    echo '</div>';
    echo '</div>';

    // Transaction table
    echo '<table class="wp-list-table widefat fixed striped posts">';
    echo '<thead>
            <tr>
                <td id="cb" class="manage-column column-cb check-column"><input type="checkbox" id="select-all"></td>
                <th>Username</th>
                <th>Type</th>
                <th>Amount</th>
                <th>Details</th>
                <th>Date</th>
                <th>Action</th>
            </tr>
          </thead>';
    echo '<tbody>';

    if (!empty($transactions)) {
        foreach ($transactions as $transaction) {
            $user = get_user_by('id', $transaction->user_id);
            $username = $user ? $user->user_login : 'Unknown';
            $date_value = str_replace(' ', 'T', substr($transaction->date, 0, 16));
            echo '<tr data-id="' . esc_attr($transaction->transaction_id) . '">';
            echo '<th scope="row" class="check-column"><input type="checkbox" name="bulk_ids[]" value="' . esc_attr($transaction->transaction_id) . '"></th>';
            echo '<td>' . esc_html($username) . '</td>';
            echo '<td><input type="text" class="wte-type" value="' . esc_attr($transaction->type) . '" style="width:100%;"></td>';
            echo '<td><input type="number" step="0.01" class="wte-amount" value="' . esc_attr($transaction->amount) . '" style="width:100%;"></td>';
            echo '<td><input type="text" class="wte-details" value="' . esc_attr($transaction->details) . '" style="width:100%;"></td>';
            echo '<td><input type="datetime-local" class="wte-date" value="' . esc_attr($date_value) . '" step="1"></td>';
            echo '<td>';
            echo '<button type="button" class="button button-primary wte-update-btn" style="margin-right:5px;">Update</button>';
            echo '<button type="button" class="button button-secondary wte-delete-btn">Delete</button>';
            echo '<span class="wte-msg" style="margin-left:10px;"></span>';
            echo '</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="7" style="text-align:center;">No transactions found.</td></tr>';
    }

    echo '</tbody>';
    echo '</table>';
    echo '</form>';

    // Pagination Links (styled like WP)
    echo '<div class="tablenav bottom" style="margin-top: 10px;">';
    echo '<div class="tablenav-pages">';
    if ($page > 1) {
        echo '<a class="button" href="?page=wallet-transactions&paged=' . ($page - 1) . '&search=' . urlencode($search_query) . '">&laquo; Previous</a> ';
    }
    echo '<span class="pagination-links"> Page ' . $page . ' of ' . $total_pages . ' </span>';
    if ($page < $total_pages) {
        echo '<a class="button" href="?page=wallet-transactions&paged=' . ($page + 1) . '&search=' . urlencode($search_query) . '">Next &raquo;</a>';
    }
    echo '</div>';
    echo '</div>';

    echo '</div>'; // End of wrap
}