'use strict';
/// <reference path="../../../js/common.d.ts" />
/// <reference path="../../../js/lodash-3.10.d.ts" />
import { AmeCustomizable } from '../../pro-customizables/assets/customizable.js';
import { registerBaseComponents } from '../../pro-customizables/ko-components/ame-components.js';
import AmeAcStructure from './ko-components/ame-ac-structure.js';
import AmeAcSection from './ko-components/ame-ac-section.js';
import AmeAcSectionLink from './ko-components/ame-ac-section-link.js';
import AmeAcControl from './ko-components/ame-ac-control.js';
import AmeAcControlGroup from './ko-components/ame-ac-control-group.js';
import AmeAcContentSection from './ko-components/ame-ac-content-section.js';
import { AmeAdminCustomizerBase } from './admin-customizer-base.js';
import AmeAcSeparator from './ko-components/ame-ac-separator.js';
import AmeAcValidationErrors from './ko-components/ame-ac-validation-errors.js';
export var AmeAdminCustomizer;
(function (AmeAdminCustomizer) {
    var SettingCollection = AmeCustomizable.SettingCollection;
    var unserializeUiElement = AmeCustomizable.unserializeUiElement;
    var unserializeSetting = AmeCustomizable.unserializeSetting;
    const $ = jQuery;
    const _ = wsAmeLodash;
    registerBaseComponents();
    ko.components.register('ame-ac-structure', AmeAcStructure);
    ko.components.register('ame-ac-section', AmeAcSection);
    ko.components.register('ame-ac-section-link', AmeAcSectionLink);
    ko.components.register('ame-ac-content-section', AmeAcContentSection);
    ko.components.register('ame-ac-control-group', AmeAcControlGroup);
    ko.components.register('ame-ac-control', AmeAcControl);
    ko.components.register('ame-ac-separator', AmeAcSeparator);
    ko.components.register('ame-ac-validation-errors', AmeAcValidationErrors);
    const reducedMotionQuery = window.matchMedia('(prefers-reduced-motion: reduce)');
    let prefersReducedMotion = reducedMotionQuery && reducedMotionQuery.matches;
    reducedMotionQuery.addEventListener('change', () => {
        prefersReducedMotion = reducedMotionQuery.matches;
    });
    class CustomizerSettingsCollection extends SettingCollection {
        constructor(ajaxUrl, saveChangesetNonce, changesetName, changesetItemCount = 0, changesetStatus = null) {
            super();
            this.ajaxUrl = ajaxUrl;
            this.saveChangesetNonce = saveChangesetNonce;
            /**
             * Settings that have changed since the last save attempt.
             */
            this.pendingSettings = {};
            /**
             * Settings that in the process of being sent to the server to be saved.
             * They might not be saved yet.
             */
            this.sentSettings = {};
            this.currentChangesetRequest = null;
            this.saveTriggerTimeoutId = null;
            this.exclusiveOperation = ko.observable(false);
            const self = this;
            this.currentChangeset = ko.observable(new Changeset(changesetName, changesetItemCount, changesetStatus));
            this.changesetName = ko.pureComputed(() => {
                var _a;
                return ((_a = self.currentChangeset()) === null || _a === void 0 ? void 0 : _a.name()) || '';
            });
            //Automatically save the changeset when any settings change.
            const totalChangeCount = ko.pureComputed(() => {
                const changeset = self.currentChangeset();
                return (changeset ? changeset.currentSessionChanges() : 0);
            });
            totalChangeCount.subscribe(_.debounce((counter) => {
                if (counter > 0) {
                    self.queueChangesetUpdate();
                }
            }, 3000, { leading: true, trailing: true }));
            this.isExclusiveOperationInProgress = ko.pureComputed(() => {
                return self.exclusiveOperation() === true;
            });
            //Keep track of unsaved changes and changesets.
            this.addChangeListener((setting) => {
                this.pendingSettings[setting.id] = setting;
                let changeset = this.currentChangeset();
                //If the current changeset cannot be modified, create a new one
                //for the changed setting(s).
                if (!(changeset === null || changeset === void 0 ? void 0 : changeset.canBeModified())) {
                    changeset = new Changeset();
                    this.currentChangeset(changeset);
                }
                //Track the number of changes in the current session.
                changeset.currentSessionChanges(changeset.currentSessionChanges() + 1);
            });
        }
        queueChangesetUpdate(delay = 0) {
            if (delay > 0) {
                if (this.saveTriggerTimeoutId !== null) {
                    //Replace the existing timeout with a new one.
                    clearTimeout(this.saveTriggerTimeoutId);
                }
                this.saveTriggerTimeoutId = setTimeout(() => {
                    this.saveTriggerTimeoutId = null;
                    this.queueChangesetUpdate(0);
                }, delay);
                return;
            }
            if (this.saveTriggerTimeoutId !== null) {
                return; //Another timeout is already waiting.
            }
            if (this.currentChangesetRequest !== null) {
                //There's an in-progress request, so wait until it's done.
                this.currentChangesetRequest.always(() => {
                    //Wait a bit to avoid hammering the server.
                    this.queueChangesetUpdate(1000);
                });
                return;
            }
            this.saveChangeset();
        }
        saveChangeset(status = null) {
            var _a;
            //Do nothing if there are no changes.
            if (_.isEmpty(this.pendingSettings) && (status === null)) {
                return $.Deferred().reject(new Error('There are no changes to save.')).promise();
            }
            if (this.isExclusiveOperationInProgress()) {
                return $.Deferred().reject(new Error('Another exclusive changeset operation is in progress.')).promise();
            }
            let isExclusiveRequest = (status === 'publish') || (status === 'trash');
            if (isExclusiveRequest) {
                this.exclusiveOperation(true);
            }
            const savedChangeset = this.currentChangeset();
            //Keep a local copy of the settings in case something changes instance
            //properties while the request is in progress (should never happen).
            const settingsToSend = this.pendingSettings;
            this.sentSettings = settingsToSend;
            this.pendingSettings = {};
            const modifiedSettings = _.mapValues(settingsToSend, setting => setting.value());
            const requestData = {
                action: 'ws_ame_ac_save_changeset',
                _ajax_nonce: this.saveChangesetNonce,
                changeset: (_a = savedChangeset === null || savedChangeset === void 0 ? void 0 : savedChangeset.name) !== null && _a !== void 0 ? _a : '',
                modified: JSON.stringify(modifiedSettings),
            };
            if (status !== null) {
                requestData['status'] = status;
            }
            //If the changeset doesn't have a name, it is new.
            if (!(savedChangeset === null || savedChangeset === void 0 ? void 0 : savedChangeset.hasName())) {
                requestData['createNew'] = 1;
            }
            const request = $.ajax({
                url: this.ajaxUrl,
                method: 'POST',
                data: requestData,
                dataType: 'json',
                timeout: 20000,
            });
            this.currentChangesetRequest = request;
            const self = this;
            function storeValidationResultsFrom(serverResponse) {
                const results = _.get(serverResponse, ['data', 'validationResults']);
                if (typeof results !== 'object') {
                    return;
                }
                for (const settingId in results) {
                    const setting = self.get(settingId);
                    if (!setting.isDefined()) {
                        continue;
                    }
                    if (!modifiedSettings.hasOwnProperty(settingId)) {
                        continue;
                    }
                    const sentValue = modifiedSettings[settingId];
                    const state = results[settingId];
                    if (state.isValid) {
                        setting.get().clearValidationErrorsForValue(sentValue);
                    }
                    else {
                        setting.get().addValidationErrorsForValue(sentValue, _.filter(state.errors, error => (typeof error.message === 'string')));
                    }
                }
            }
            function storeChangesetDetailsFrom(serverResponse) {
                if (!savedChangeset) {
                    return;
                }
                //Store the returned changeset name in case a new changeset was created.
                if (!savedChangeset.hasName()) {
                    const newName = _.get(serverResponse, ['data', 'changeset']);
                    if (typeof newName === 'string') {
                        savedChangeset.name(newName);
                    }
                }
                //Store the changeset status.
                const newStatus = _.get(serverResponse, ['data', 'changesetStatus']);
                if (typeof newStatus === 'string') {
                    savedChangeset.status(newStatus);
                }
                //Store the number of changes in the changeset.
                const newChangeCount = _.get(serverResponse, ['data', 'changesetItemCount']);
                if (typeof newChangeCount === 'number') {
                    savedChangeset.knownItemCount(newChangeCount);
                }
                //Was the changeset published? Because changesets are typically moved
                //to trash after publishing, "status" might be "trash" instead of "publish",
                //but we still want to know if it was successfully published.
                const wasPublished = _.get(serverResponse, ['data', 'changesetWasPublished'], null);
                if (wasPublished) {
                    savedChangeset.wasPublished(wasPublished);
                }
            }
            request.done(function (response) {
                storeChangesetDetailsFrom(response);
                storeValidationResultsFrom(response);
                //After successfully publishing a changeset, it has no more
                //unsaved changes.
                const isPublished = (savedChangeset.status() === 'publish')
                    || (savedChangeset.status() === 'future')
                    || (savedChangeset.wasPublished());
                if (isPublished) {
                    savedChangeset.currentSessionChanges(0);
                }
                //After a changeset is published or trashed, it can no longer
                //be edited. We may be able to replace it with a new changeset
                //that was created on the server.
                if (!self.currentChangeset().canBeModified()) {
                    const nextChangeset = _.get(response, ['data', 'nextChangeset']);
                    if ((typeof nextChangeset === 'string') && (nextChangeset !== '')) {
                        self.currentChangeset(new Changeset(nextChangeset));
                    }
                }
            });
            request.fail((requestObject) => {
                if (typeof requestObject.responseJSON === 'object') {
                    storeValidationResultsFrom(requestObject.responseJSON);
                    storeChangesetDetailsFrom(requestObject.responseJSON);
                }
                //Add the unsaved settings back to the pending list.
                for (const id in settingsToSend) {
                    //Keep only settings that still exist.
                    if (this.get(id).isDefined()) {
                        this.pendingSettings[id] = settingsToSend[id];
                    }
                }
                //We don't automatically retry because the problem might be something
                //that doesn't get better on its own, like missing permissions.
            });
            request.always(() => {
                this.currentChangesetRequest = null;
                this.sentSettings = {};
                if (isExclusiveRequest) {
                    this.exclusiveOperation(false);
                }
            });
            return request;
        }
        getCurrentChangeset() {
            return this.currentChangeset();
        }
        /**
         * Get any unsaved setting changes.
         *
         * @returns An object mapping setting IDs to their modified values.
         */
        get unsavedChanges() {
            //Include both pending settings and sent settings. Sent settings
            //might not be saved yet.
            let unsavedSettings = {};
            _.defaults(unsavedSettings, this.pendingSettings, this.sentSettings);
            return _.mapValues(unsavedSettings, setting => setting.value());
        }
        publishChangeset() {
            if (this.isExclusiveOperationInProgress()) {
                return $.Deferred()
                    .reject(new Error('Another exclusive changeset operation is already in progress.'))
                    .promise();
            }
            return this.saveChangeset('publish');
        }
    }
    class Changeset {
        constructor(name = '', knownItemCount = 0, initialStatus = '') {
            /**
             * The number of times settings have been changed in this changeset
             * during the current customizer session.
             *
             * Note that this is not the same as the number settings in the changeset:
             * if the same setting is changed X times, this counter will increase by X,
             * but the changeset will still only have one entry for that setting.
             */
            this.currentSessionChanges = ko.observable(0);
            /**
             * Once a changeset has been published or deleted, its contents can't be modified any more.
             * @private
             */
            this.fixedContentStatuses = { 'publish': true, 'trash': true, 'future': true };
            this.wasPublished = ko.observable(false);
            this.name = ko.observable(name);
            this.knownItemCount = ko.observable(knownItemCount);
            this.status = ko.observable(initialStatus !== null && initialStatus !== void 0 ? initialStatus : '');
        }
        hasName() {
            const name = this.name();
            return (typeof name === 'string') && (name !== '');
        }
        canBeModified() {
            return !this.fixedContentStatuses.hasOwnProperty(this.status());
        }
        isNonEmpty() {
            return (this.currentSessionChanges() > 0) || (this.knownItemCount() > 0);
        }
    }
    class SectionNavigation {
        constructor() {
            this.sectionNavStack = ko.observableArray([]);
            this.$sectionList = null;
            this.$sectionList = $('#ame-ac-container-collection');
            this.$sectionList.on('click', '.ame-ac-section-link', (event) => {
                event.preventDefault();
                const targetId = $(event.currentTarget).data('target-id');
                if (targetId) {
                    this.navigateToSection(targetId);
                }
            });
            this.$sectionList.on('click', '.ame-ac-section-back-button', (event) => {
                event.preventDefault();
                this.navigateBack();
            });
            this.breadcrumbs = ko.pureComputed(() => {
                return this.sectionNavStack()
                    .map((sectionId) => $('#' + sectionId))
                    .filter(($section) => $section.length > 0)
                    .map(($section) => {
                    return {
                        title: $section.find('.ame-ac-section-title .ame-ac-section-own-title')
                            .first().text()
                    };
                });
            });
        }
        navigateToSection(sectionElementId) {
            const $section = $('#' + sectionElementId);
            if ($section.length === 0) {
                return;
            }
            if ($section.hasClass('ame-ac-current-section')) {
                return; //Already on this section.
            }
            //If the requested section is in the navigation stack, navigate back
            //to it instead of putting more sections on the stack.
            const stackIndex = this.sectionNavStack.indexOf(sectionElementId);
            if (stackIndex !== -1) {
                while (this.sectionNavStack().length > stackIndex) {
                    this.navigateBack();
                }
                return;
            }
            const $previousSection = this.$sectionList.find('.ame-ac-current-section');
            if ($previousSection.length > 0) {
                this.expectTransition($previousSection, '.ame-ac-section');
                $previousSection
                    .removeClass('ame-ac-current-section')
                    .addClass('ame-ac-previous-section');
                this.sectionNavStack.push($previousSection.attr('id'));
                $previousSection.trigger('adminMenuEditor:leaveSection');
            }
            this.expectTransition($section, '.ame-ac-section');
            $section.addClass('ame-ac-current-section');
            $section.trigger('adminMenuEditor:enterSection');
        }
        navigateBack() {
            if (this.sectionNavStack().length < 1) {
                return;
            }
            const $newCurrentSection = $('#' + this.sectionNavStack.pop());
            if ($newCurrentSection.length === 0) {
                return;
            }
            const $oldCurrentSection = this.$sectionList.find('.ame-ac-current-section');
            this.expectTransition($oldCurrentSection, '.ame-ac-section');
            $oldCurrentSection.removeClass('ame-ac-current-section ame-ac-previous-section');
            $oldCurrentSection.trigger('adminMenuEditor:leaveSection');
            const $oldPreviousSection = this.$sectionList.find('.ame-ac-previous-section');
            $oldPreviousSection.removeClass('ame-ac-previous-section');
            //Show the new current section.
            this.expectTransition($newCurrentSection, '.ame-ac-section');
            $newCurrentSection.addClass('ame-ac-current-section');
            $newCurrentSection.trigger('adminMenuEditor:enterSection');
            //The next section in the stack becomes the previous section.
            if (this.sectionNavStack().length > 0) {
                this.$sectionList.find('#' + this.sectionNavStack()[this.sectionNavStack().length - 1])
                    .addClass('ame-ac-previous-section');
            }
        }
        //Add a special class to sections when they have an active CSS transition.
        //This is used to keep both sections visible while the previous section
        //slides out and the next section slides in.
        expectTransition($element, requiredSelector) {
            if (prefersReducedMotion) {
                return;
            }
            if ($element.data('ameHasTransitionEvents')) {
                return; //Event handler(s) already added.
            }
            const transitionEvents = 'transitionend transitioncancel';
            $element.addClass('ame-ac-transitioning');
            function transitionEndCallback(event) {
                //Ignore events that bubble from child elements.
                if (!$(event.target).is(requiredSelector)) {
                    return;
                }
                $element
                    .off(transitionEvents, transitionEndCallback)
                    .data('ameHasTransitionEvents', null)
                    .removeClass('ame-ac-transitioning');
            }
            $element.data('ameHasTransitionEvents', true);
            $element.on(transitionEvents, transitionEndCallback);
        }
    }
    class AdminCustomizer extends AmeAdminCustomizerBase.AdminCustomizerBase {
        constructor(scriptData) {
            super(scriptData);
            this.exitPromptMessage = 'Unsaved changes will be lost if you navigate away from this page.';
            this.$previewFrame = null;
            /**
             * Preview frame URL.
             */
            this.currentPreviewUrl = null;
            this.previewConnection = null;
            this.$saveButton = null;
            this._isFrameLoading = false;
            this.frameLoadingTimeoutId = null;
            this.lastPreviewLoadTimestamp = new Date(0);
            this.reloadWaitTimeoutId = null;
            this.hasPendingPreviewReload = false;
            this.settings = new CustomizerSettingsCollection(scriptData.ajaxUrl, scriptData.saveChangesetNonce, scriptData.changesetName, scriptData.changesetItemCount, scriptData.changesetStatus);
            _.forOwn(scriptData.settings, (data, id) => {
                this.settings.add(unserializeSetting(id, data));
            });
            let sectionIdCounter = 0;
            this.interfaceStructure = unserializeUiElement(scriptData.interfaceStructure, this.settings.get.bind(this.settings), (data) => {
                switch (data.t) {
                    case 'section':
                        data.component = 'ame-ac-section';
                        //All sections must have unique IDs for navigation to work.
                        if (!data.id) {
                            data.id = 'autoID-' + (++sectionIdCounter);
                        }
                        break;
                    case 'control-group':
                        data.component = 'ame-ac-control-group';
                        break;
                    case 'control':
                        //Tell controls that use number inputs to position the popup
                        //slider within the customizer sidebar.
                        if ((data.component === 'ame-number-input')
                            || (data.component === 'ame-box-dimensions')) {
                            data.params = data.params || {};
                            data.params.popupSliderWithin = '#ame-ac-sidebar-content';
                        }
                        //Replace regular separators with AC-specific ones.
                        if (data.component === 'ame-horizontal-separator') {
                            data.component = 'ame-ac-separator';
                        }
                }
            });
            //Add the changeset name to the URL (if not already present).
            const currentUrl = new URL(window.location.href);
            if (currentUrl.searchParams.get('ame-ac-changeset') !== this.settings.changesetName()) {
                currentUrl.searchParams.set('ame-ac-changeset', this.settings.changesetName());
                window.history.replaceState({}, '', currentUrl.href);
            }
            //When the changeset name changes, also change the URL. Discourage navigating
            //to the old URL (no pushState()) because the name is only expected to change
            //when the old changeset becomes invalid (e.g. it's deleted or published).
            this.settings.changesetName.subscribe((changesetName) => {
                const url = new URL(window.location.href);
                url.searchParams.set('ame-ac-changeset', changesetName);
                window.history.replaceState({}, '', url.href);
            });
            this.$saveButton = $('#ame-ac-apply-changes');
            //The save button should be enabled when:
            // - There are non-zero changes in the current changeset.
            // - All settings are valid.
            // - The changeset is not in the process of being published, deleted, etc.
            // - The contents of the changeset can be modified (e.g. not already published).
            const isSaveButtonEnabled = ko.pureComputed(() => {
                const changeset = this.settings.getCurrentChangeset();
                return (changeset.isNonEmpty()
                    && changeset.canBeModified()
                    && !this.settings.isExclusiveOperationInProgress()
                    && !this.settings.hasValidationErrors());
            });
            //Update button state when the customizer loads.
            this.$saveButton.prop('disabled', !isSaveButtonEnabled());
            //And also on changes.
            isSaveButtonEnabled.subscribe((isEnabled) => {
                var _a;
                this.$saveButton.prop('disabled', !isEnabled);
                //Change the text back to the default when the button is enabled.
                if (isEnabled) {
                    this.$saveButton.val((_a = this.$saveButton.data('default-text')) !== null && _a !== void 0 ? _a : 'Save Changes');
                }
            });
            //Handle the "Save Changes" button.
            this.$saveButton.on('click', () => {
                //Show the spinner.
                const $spinner = $('#ame-ac-primary-actions .spinner');
                $spinner.css('visibility', 'visible').show();
                const publishFailNoticeId = 'ame-ac-publish-failed-notice';
                //Remove the previous error notification, if any.
                $('#' + publishFailNoticeId).remove();
                const promise = this.settings.publishChangeset();
                promise.fail((error) => {
                    //Show a dismissible error notification.
                    let message = 'An unexpected error occurred while saving changes.';
                    if (typeof error === 'string') {
                        message = error;
                    }
                    else if (error instanceof Error) {
                        message = error.message;
                    }
                    else if (typeof error.responseJSON === 'object') {
                        message = _.get(error.responseJSON, ['data', 'message'], message);
                    }
                    const $notice = $('<div>')
                        .attr('id', publishFailNoticeId)
                        .addClass('notice notice-error is-dismissible')
                        .text(message);
                    //WordPress won't automatically add the dismiss button to a dynamically
                    //generated notice like this, so we have to do it.
                    $notice.append($('<button type="button" class="notice-dismiss"></button>')
                        .append('<span class="screen-reader-text">Dismiss this notice</span>')
                        .on('click', (event) => {
                        event.preventDefault();
                        $notice.remove(); //Not as fancy as WP does it.
                    }));
                    const $container = $('#ame-ac-global-notification-area');
                    $container.append($notice);
                });
                promise.done(() => {
                    var _a;
                    this.$saveButton.val((_a = this.$saveButton.data('published-text')) !== null && _a !== void 0 ? _a : 'Saved');
                    //The preview could be stale. For example, the color scheme module
                    //switches between "actual" and "preview" color schemes dynamically,
                    //but the "actual" scheme could change after applying new settings.
                    //Let's reload the preview frame to make sure it's up-to-date.
                    this.queuePreviewFrameReload();
                });
                promise.always(() => {
                    $spinner.css('visibility', 'hidden');
                });
            });
            //Prevent the user from interacting with settings while the changeset is being modified.
            this.settings.isExclusiveOperationInProgress.subscribe((isInProgress) => {
                $('#ame-ac-sidebar-blocker-overlay').toggle(isInProgress);
            });
            this.sectionNavigation = new SectionNavigation();
            //Set up the preview frame.
            this.$previewFrame = $('iframe#ame-ac-preview');
            this.initialPreviewUrl = scriptData.initialPreviewUrl;
            this.refreshPreviewNonce = scriptData.refreshPreviewNonce;
            this.$previewFrame.on('load', () => {
                this.isFrameLoading = false;
                //The URL that was actually loaded might not match the one that
                //was requested (e.g. because there was a redirect).
                this.currentPreviewUrl = null;
                //Close the previous postMessage connection.
                if (this.previewConnection) {
                    this.previewConnection.disconnect();
                    this.previewConnection = null;
                }
                const frame = this.$previewFrame.get(0);
                if (!frame || !(frame instanceof HTMLIFrameElement)) {
                    return;
                }
                //Try to get the preview URL from the iframe.
                try {
                    const url = frame.contentWindow.location.href;
                    if (url) {
                        this.currentPreviewUrl = url;
                    }
                }
                catch (e) {
                    //We can't get the URL directly, probably because it's a cross-origin iframe.
                }
                this.previewConnection = AmeAcCommunicator.connectToChild(frame, {
                    'setPreviewUrl': (url) => {
                        if (this.isPreviewableUrl(url)) {
                            this.previewUrl = url;
                        }
                    },
                    'notifyPreviewUrlChanged': (url) => {
                        this.currentPreviewUrl = url;
                    }
                }, this.allowedCommOrigins, scriptData.isWpDebugEnabled);
                this.previewConnection.promise.then((connection) => {
                    connection.execute('getCurrentUrl').then((url) => {
                        if (url && (typeof url === 'string')) {
                            this.currentPreviewUrl = url;
                        }
                    });
                    //Notify other scripts that the preview frame is loaded and
                    //the postMessage connection is ready for use.
                    $('body').trigger('adminMenuEditor:acPreviewConnectionReady');
                });
            });
            this.previewUrl = this.initialPreviewUrl;
            //Notify other scripts. This lets them register custom controls and so on.
            $('#ame-ac-admin-customizer').trigger('adminMenuEditor:acRegister', [this]);
            const throttledReloadPreview = _.throttle(() => {
                this.queuePreviewFrameReload();
            }, 1000, //The reload method does its own throttling, so we use a low wait time here.
            { leading: true, trailing: true });
            //Refresh the preview when any setting changes.
            this.settings.addChangeListener((setting, newValue) => {
                if (setting.supportsPostMessage
                    && this.previewConnection
                    && this.previewConnection.isConnected) {
                    this.previewConnection.execute('previewSetting', setting.id, newValue);
                }
                else {
                    throttledReloadPreview();
                }
            });
            const registerUnloadPrompt = () => {
                //Ask for confirmation when the user tries to leave the page and the changeset
                //has unpublished changes.
                $(window).on('beforeunload.ame-ac-exit-confirm', (event) => {
                    if (this.hasUnpublishedChanges()) {
                        event.preventDefault();
                        //Note: The confirmation prompt will only be displayed if the user
                        //has interacted with the page (e.g. clicked something).
                        //As of this writing, MDN says that some browsers still don't support triggering
                        //an "unsaved changes" prompt with event.preventDefault(). You need to set
                        //event.returnValue to a string or return a string from the event handler.
                        //Modern browsers will ignore the content and display their own generic message.
                        return this.exitPromptMessage;
                    }
                });
            };
            /*
             Allegedly, registering a beforeunload handler can cause the browser to
             disable some optimizations, so let's only do it when the user changes
             something or the changeset already contains some changes.
             */
            if (this.settings.getCurrentChangeset().isNonEmpty()) {
                registerUnloadPrompt();
            }
            else {
                const listenerId = this.settings.addChangeListener(() => {
                    //Remove the listener after it has been triggered once.
                    this.settings.removeChangeListener(listenerId);
                    registerUnloadPrompt();
                });
            }
        }
        getSettingObservable(settingId, defaultValue) {
            //Let's just implement this temporarily while working on refactoring this
            //stuff to use KO components.
            return this.settings
                .get(settingId)
                .map(setting => setting.value)
                .getOrElse(ko.observable(defaultValue));
        }
        getAllSettingValues() {
            throw new Error('Method not implemented.');
        }
        get previewUrl() {
            return this.currentPreviewUrl;
        }
        set previewUrl(url) {
            if (url === this.currentPreviewUrl) {
                return;
            }
            if (this.isPreviewableUrl(url)) {
                this.navigatePreviewFrame(url);
            }
        }
        navigatePreviewFrame(url = null, forceReload = false) {
            const oldUrl = this.previewUrl;
            if (url === null) {
                url = oldUrl !== null && oldUrl !== void 0 ? oldUrl : this.initialPreviewUrl;
            }
            const isSameUrl = (oldUrl === url);
            if (isSameUrl && !forceReload) {
                return;
            }
            //If there are any unsaved changes, let's include them in the preview by simulating
            //a form submission and sending the changes as form data. The server-side component
            //will merge these changes with existing changeset data.
            const unsavedChanges = this.settings.unsavedChanges;
            const simulateFormSubmission = !_.isEmpty(unsavedChanges);
            const parsedUrl = new URL(url);
            //If we're not using form submission, add a special parameter
            //to the URL to force a refresh.
            const refreshParam = '_ame-ac-refresh-trigger';
            if (isSameUrl && !simulateFormSubmission) {
                parsedUrl.searchParams.set(refreshParam, Date.now() + '_' + Math.random());
            }
            else {
                //Otherwise, remove the parameter just to be safe.
                parsedUrl.searchParams.delete(refreshParam);
            }
            //Ensure that the changeset used in the preview matches the current
            //changeset and preview is enabled. This is just a precaution. Normally,
            //the preview script automatically changes link URLs.
            parsedUrl.searchParams.set('ame-ac-changeset', this.settings.changesetName());
            parsedUrl.searchParams.set('ame-ac-preview', '1');
            this.hasPendingPreviewReload = false; //Reloading now, so no longer pending.
            this.isFrameLoading = true;
            //console.info('navigatePreviewFrame: Navigating to ' + parsedUrl.href);
            if (simulateFormSubmission) {
                const formData = {
                    action: 'ws_ame_ac_refresh_preview_frame',
                    "ame-ac-changeset": this.settings.changesetName(),
                    modified: JSON.stringify(unsavedChanges),
                    nonce: this.refreshPreviewNonce
                };
                const $form = $('<form>')
                    .attr('method', 'post')
                    .attr('action', parsedUrl.href)
                    .attr('target', 'ame-ac-preview-frame')
                    .appendTo('body');
                for (const key in formData) {
                    const value = formData[key];
                    $('<input>')
                        .attr('type', 'hidden')
                        .attr('name', key)
                        .val(value)
                        .appendTo($form);
                }
                this.currentPreviewUrl = parsedUrl.href;
                $form.trigger('submit');
                $form.remove();
            }
            else {
                this.currentPreviewUrl = parsedUrl.href;
                this.$previewFrame.attr('src', this.currentPreviewUrl);
            }
        }
        set isFrameLoading(isLoading) {
            const wasLoadingBefore = this._isFrameLoading;
            if (!isLoading && (isLoading === wasLoadingBefore)) {
                return;
            }
            //In some circumstances, we may start to load a new URL before
            //the previous one has finished loading. This is valid and should
            //reset the load timeout.
            $('#ame-ac-preview-refresh-indicator').toggleClass('ame-ac-show-indicator', isLoading);
            if (this.frameLoadingTimeoutId) {
                clearTimeout(this.frameLoadingTimeoutId);
                this.frameLoadingTimeoutId = null;
            }
            if (isLoading) {
                //As a precaution, we'll assume that if the frame doesn't load in a reasonable
                //time, it will never finish loading.
                this.frameLoadingTimeoutId = window.setTimeout(() => {
                    if (this.isFrameLoading) {
                        this.isFrameLoading = false;
                    }
                }, 20000);
            }
            this._isFrameLoading = isLoading;
            if (wasLoadingBefore && !isLoading) {
                this.lastPreviewLoadTimestamp = new Date();
            }
            //Once the frame is loaded, trigger any pending reload.
            if (!isLoading && this.hasPendingPreviewReload) {
                this.hasPendingPreviewReload = false;
                this.queuePreviewFrameReload();
            }
        }
        get isFrameLoading() {
            return this._isFrameLoading;
        }
        queuePreviewFrameReload() {
            if (this.reloadWaitTimeoutId) {
                return; //The frame will reload soon.
            }
            if (this.isFrameLoading) {
                this.hasPendingPreviewReload = true;
                return;
            }
            //To avoid stressing the server, wait at least X ms after the last
            //load completes before reloading the frame.
            const reloadWaitTime = 2000;
            const now = new Date();
            const timeSinceLastLoad = now.getTime() - this.lastPreviewLoadTimestamp.getTime();
            if (timeSinceLastLoad < reloadWaitTime) {
                this.reloadWaitTimeoutId = window.setTimeout(() => {
                    this.reloadWaitTimeoutId = null;
                    this.queuePreviewFrameReload();
                }, reloadWaitTime - timeSinceLastLoad);
                return;
            }
            //Actually reload the frame.
            this.navigatePreviewFrame(null, true);
        }
        navigateToRootSection() {
            this.sectionNavigation.navigateToSection('ame-ac-section-structure-root');
        }
        // noinspection JSUnusedGlobalSymbols -- Used in at least one add-on.
        /**
         * Execute an RPC method in the preview frame.
         *
         * @param {string} methodName
         * @param {*} args
         */
        executeRpcMethod(methodName, ...args) {
            if (!this.previewConnection || !this.previewConnection.isConnected) {
                return $.Deferred().reject('The preview frame is not connected.').promise();
            }
            return this.previewConnection.execute(methodName, ...args);
        }
        confirmExit() {
            if (this.hasUnpublishedChanges()) {
                if (window.confirm(this.exitPromptMessage)) {
                    //Remove the confirmation prompt that appears when leaving the page.
                    //We don't want to show two prompts.
                    $(window).off('beforeunload.ame-ac-exit-confirm');
                    return true;
                }
                return false;
            }
            return true;
        }
        hasUnpublishedChanges() {
            const changeset = this.settings.getCurrentChangeset();
            return changeset.isNonEmpty() && !changeset.wasPublished();
        }
    }
    AmeAdminCustomizer.AdminCustomizer = AdminCustomizer;
})(AmeAdminCustomizer || (AmeAdminCustomizer = {}));
jQuery(function () {
    //Give other scripts a chance to load before we start.
    //Some of them also use jQuery to run when the DOM is ready.
    setTimeout(() => {
        window['wsAdminCustomizer'] = new AmeAdminCustomizer.AdminCustomizer(wsAmeAdminCustomizerData);
        ko.applyBindings(window['wsAdminCustomizer'], document.getElementById('ame-ac-admin-customizer'));
        //Navigate to the root section. In the current implementation this can't happen
        //until bindings have been applied, so it's not part of the constructor.
        setTimeout(() => {
            window['wsAdminCustomizer'].navigateToRootSection();
        }, 5); //Components are rendered asynchronously.
    }, 20);
});
//# sourceMappingURL=admin-customizer.js.map